<?php
/**
 * Search and reaplace manager
 *
 * Standard: PSR-2
 * @link http://www.php-fig.org/psr/psr-2 Full Documentation
 *
 * @package SC\DUPX\U
 *
 */

/**
 * Search and replace manager
 * singleton class
 */
final class DUPX_S_R_MANAGER
{
    const GLOBAL_SCOPE_KEY = '___!GLOBAL!___!SCOPE!___';

    /**
     *
     * @var DUPX_S_R_MANAGER
     */
    private static $instance = null;

    /**
     * full list items not sorted
     * @var DUPX_S_R_ITEM[]
     */
    private $items = array();

    /**
     * items sorted by priority and scope
     * [
     *      10 => [
     *             '___!GLOBAL!___!SCOPE!___' => [
     *                  DUPX_S_R_ITEM
     *                  DUPX_S_R_ITEM
     *                  DUPX_S_R_ITEM
     *              ],
     *              'scope_one' => [
     *                  DUPX_S_R_ITEM
     *                  DUPX_S_R_ITEM
     *              ]
     *          ],
     *      20 => [
     *          .
     *          .
     *          .
     *      ]
     * ]
     *
     * @var array
     */
    private $prorityScopeItems = array();

    /**
     *
     * @return DUPX_S_R_MANAGER
     */
    public static function getInstance()
    {
        if (is_null(self::$instance)) {
            self::$instance = new self();
        }

        return self::$instance;
    }

    private function __construct()
    {

    }

    /**
     *
     * @param string $search
     * @param string $replace
     * @param string $type                  // item type DUPX_S_R_ITEM::[TYPE_STRING|TYPE_URL|TYPE_PATH]
     * @param int $prority                  // lower first
     * @param bool|string|string[] $scope   // true = global scope | false = never | string signle scope | string[] scope list
     *
     * @return boolean|DUPX_S_R_ITEM        // false if fail or new DUPX_S_R_ITEM
     */
    public function addItem($search, $replace, $type = DUPX_S_R_ITEM::TYPE_STRING, $prority = 10, $scope = true)
    {
        if (strlen((string) $search) == 0) {
            return false;
        }

        if (is_bool($scope)) {
            $scope = $scope ? self::GLOBAL_SCOPE_KEY : '';
        }
        DUPX_Log::info(
            'ADD SEARCH AND REPLACE ITEM'."\n".
            'Search:"'.$search.'" Replace:"'.$replace.'" Type:"'.$type.'" Prority:"'.$prority.'" Scope:"'.$scope, 2);
        $new_item = new DUPX_S_R_ITEM($search, $replace, $type, $prority, $scope);

        $this->items[$new_item->getId()] = $new_item;

        // create priority array
        if (!isset($this->prorityScopeItems[$new_item->prority])) {
            $this->prorityScopeItems[$new_item->prority] = array();

            // sort by priority
            ksort($this->prorityScopeItems);
        }

        // create scope list
        foreach ($new_item->scope as $scope) {
            if (!isset($this->prorityScopeItems[$new_item->prority][$scope])) {
                $this->prorityScopeItems[$new_item->prority][$scope] = array();
            }
            $this->prorityScopeItems[$new_item->prority][$scope][] = $new_item;
        }

        return $new_item;
    }

    /**
     * get all search and reaple items by scpoe
     *
     * @param null|string $scope if scope is empty get only global scope
     * @return DUPX_S_R_ITEM[]
     */
    private function getSearchReplaceItems($scope = null, $globalScope = true)
    {
        $items_list = array();
        foreach ($this->prorityScopeItems as $priority => $priority_list) {
            // get scope list
            if (!empty($scope) && isset($priority_list[$scope])) {
                foreach ($priority_list[$scope] as $item) {
                    $items_list[] = $item;
                }
            }

            // get global scope
            if ($globalScope && isset($priority_list[self::GLOBAL_SCOPE_KEY])) {
                foreach ($priority_list[self::GLOBAL_SCOPE_KEY] as $item) {
                    $items_list[] = $item;
                }
            }
        }

        return $items_list;
    }

    /**
     * get replace list by scope
     * result
     * [
     *      ['search' => ...,'replace' => ...]
     *      ['search' => ...,'replace' => ...]
     * ]
     *
     * @param null|string $scope if scope is empty get only global scope
     * @param bool $unique_search If true it eliminates the double searches leaving the one with lower priority.
     *
     * @return array
     */
    public function getSearchReplaceList($scope = null, $unique_search = true, $globalScope = true)
    {
        DUPX_Log::info(
            '-- SEARCH LIST  --'."\n".
            '   SCOPE: "'.(string) $scope.'"', 2);

        $items_list = $this->getSearchReplaceItems($scope, $globalScope);
        DUPX_Log::info(
            '-- SEARCH LIST ITEMS --'."\n".
            print_r($items_list, true), 3);

        if ($unique_search) {
            $items_list = self::uniqueSearchListItem($items_list);
            DUPX_Log::info(
                '-- UNIQUE LIST ITEMS --'."\n".
                print_r($items_list, true), 3);
        }
        $result = array();

        foreach ($items_list as $item) {
            $result = array_merge($result, $item->getPairsSearchReplace());
        }

        if ($GLOBALS["LOGGING"] >= 1) {
            $log_str = '';
            foreach ($result as $index => $c_sr) {
                $sIndex  = $index + 1;
                $log_str .= 'Search'.str_pad($sIndex, 3, ' ', STR_PAD_LEFT).":".
                    str_pad("\"".$c_sr['search']."\" ", 50, '=', STR_PAD_RIGHT).
                    "=> \"".$c_sr['replace']."\"\n";
            }
            DUPX_Log::info(
                $log_str.
                '-----------------', 1);
        }
        return $result;
    }

    /**
     * remove duplicated search strings. 
     * Leave the object at lower priority
     *
     * @param DUPX_S_R_ITEM[] $list
     * @return boolean|DUPX_S_R_ITEM[]
     */
    private static function uniqueSearchListItem($list)
    {
        $search_strings = array();
        $result         = array();

        if (!is_array($list)) {
            return false;
        }

        foreach ($list as $item) {
            if (!in_array($item->search, $search_strings)) {
                $result[]         = $item;
                $search_strings[] = $item->search;
            }
        }

        return $result;
    }

    private function __clone()
    {

    }

    private function __wakeup()
    {

    }
}

/**
 * search and replace item use in manager to creat the search and replace list.
 */
class DUPX_S_R_ITEM
{
    private static $uniqueIdCount = 0;

    const TYPE_STRING = 'str';
    const TYPE_URL    = 'url';
    const TYPE_PATH   = 'path';

    /**
     *
     * @var int
     */
    private $id = 0;

    /**
     *
     * @var int prority lower first
     */
    public $prority = 10;

    /**
     *
     * @var string[] scope list
     */
    public $scope = array();

    /**
     *
     * @var string type of string
     */
    public $type = self::TYPE_STRING;

    /**
     *
     * @var string search string
     */
    public $search = '';

    /**
     *
     * @var string replace string
     */
    public $replace = '';

    /**
     *
     * @param string $search
     * @param string $replace
     * @param string $type
     * @param int $prority
     * @param string|string[] $scope if empty never used
     */
    public function __construct($search, $replace, $type = DUPX_S_R_ITEM::TYPE_STRING, $prority = 10, $scope = array())
    {
        if (!is_array($scope)) {
            $this->scope = empty($scope) ? array() : array((string) $scope);
        } else {
            $this->scope = $scope;
        }
        $this->prority = (int) $prority;
        $this->search  = $type == DUPX_S_R_ITEM::TYPE_URL ? rtrim($search, '/') : (string) $search;
        $this->replace = $type == DUPX_S_R_ITEM::TYPE_URL ? rtrim($replace, '/') : (string) $replace;
        $this->type    = $type;
        $this->id      = self::$uniqueIdCount;
        self::$uniqueIdCount ++;
    }

    /**
     * return search an replace string
     *
     * result
     * [
     *      ['search' => ...,'replace' => ...]
     *      ['search' => ...,'replace' => ...]
     * ]
     *
     * @return array
     */
    public function getPairsSearchReplace()
    {
        switch ($this->type) {
            case self::TYPE_URL:
                return self::searchReplaceUrl($this->search, $this->replace);
            case self::TYPE_PATH:
                return self::searchReplacePath($this->search, $this->replace);
            case self::TYPE_STRING:
            default:
                return self::searchReplaceWithEncodings($this->search, $this->replace);
        }
    }

    /**
     * Get search and replace strings with encodings
     * prevents unnecessary substitution like when search and reaplace are the same.
     *
     * result
     * [
     *      ['search' => ...,'replace' => ...]
     *      ['search' => ...,'replace' => ...]
     * ]
     *
     * @param string $search
     * @param string $replace
     * @param bool $json add json encode string
     * @param bool $urlencode add urlencode string
     *
     * @return array pairs search and replace
     */
    public static function searchReplaceWithEncodings($search, $replace, $json = true, $urlencode = true)
    {
        $result = array();
        if ($search != $replace) {
            $result[] = array('search' => $search, 'replace' => $replace);
        } else {
            return array();
        }

        // JSON ENCODE
        if ($json) {
            $search_json  = str_replace('"', "", json_encode($search));
            $replace_json = str_replace('"', "", json_encode($replace));

            if ($search != $search_json && $search_json != $replace_json) {
                $result[] = array('search' => $search_json, 'replace' => $replace_json);
            }
        }

        // URL ENCODE
        if ($urlencode) {
            $search_urlencode  = urlencode($search);
            $replace_urlencode = urlencode($replace);

            if ($search != $search_urlencode && $search_urlencode != $replace_urlencode) {
                $result[] = array('search' => $search_urlencode, 'replace' => $replace_urlencode);
            }
        }

        return $result;
    }

    /**
     * Add replace strings to substitute old url to new url
     * 1) no protocol old url to no protocol new url (es. //www.hold.url  => //www.new.url)
     * 2) wrong protocol new url to right protocol new url (es. http://www.new.url => https://www.new.url)
     * 
     * result
     * [
     *      ['search' => ...,'replace' => ...]
     *      ['search' => ...,'replace' => ...]
     * ]
     *
     * @param string $search_url
     * @param string $replace_url
     * @param bool $force_new_protocol if true force http or https protocol (work only if replace url have http or https scheme)
     *
     * @return array
     */
    public static function searchReplaceUrl($search_url, $replace_url, $force_new_protocol = true)
    {
        if (($parse_search_url = parse_url($search_url)) !== false && isset($parse_search_url['scheme'])) {
            $search_url_raw = substr($search_url, strlen($parse_search_url['scheme']) + 1);
        } else {
            $search_url_raw = $search_url;
        }

        if (($parse_replace_url = parse_url($replace_url)) !== false && isset($parse_replace_url['scheme'])) {
            $replace_url_raw = substr($replace_url, strlen($parse_replace_url['scheme']) + 1);
        } else {
            $replace_url_raw = $replace_url;
        }

        //SEARCH WITH NO PROTOCOL: RAW "//"
        $result = self::searchReplaceWithEncodings($search_url_raw, $replace_url_raw);


        if ($force_new_protocol && $parse_replace_url !== false && isset($parse_replace_url['scheme'])) {
            //FORCE NEW PROTOCOL [HTTP / HTTPS]
            switch ($parse_replace_url['scheme']) {
                case 'http':
                    $replace_url_wrong_protocol = 'https:'.$replace_url_raw;
                    break;
                case 'https':
                    $replace_url_wrong_protocol = 'http:'.$replace_url_raw;
                    break;
                default:
                    $replace_url_wrong_protocol = '';
                    break;
            }

            if (!empty($replace_url_wrong_protocol)) {
                $result = array_merge($result, self::searchReplaceWithEncodings($replace_url_wrong_protocol, $replace_url));
            }
        }

        return $result;
    }

    /**
     * result
     * [
     *      ['search' => ...,'replace' => ...]
     *      ['search' => ...,'replace' => ...]
     * ]
     *
     * @param string $search_path
     * @param string $replace_path
     * 
     * @return array
     */
    public static function searchReplacePath($search_path, $replace_path)
    {
        $result = self::searchReplaceWithEncodings($search_path, $replace_path);

        $search_path_unsetSafe  = rtrim(DUPX_U::unsetSafePath($search_path), '\\');
        $replace_path_unsetSafe = rtrim($replace_path, '/');
        $result                 = array_merge($result, self::searchReplaceWithEncodings($search_path_unsetSafe, $replace_path_unsetSafe));

        return $result;
    }

    /**
     * get unique item id
     *
     * @return int
     */
    public function getId()
    {
        return $this->id;
    }
}